;-----------------------------------------------------------------------------
;  Copyright (C) 2002 Peter Millett
;  All rights reserved.
;
;
;
;  FILE NAME   :  813_1_1.ASM 
;  TARGET MCU  :  C8051F206 
;  DESCRIPTION :  This code is to run the power supply controller for the 813
;                 SE tube amplifier.
;  
; 
;
;-----------------------------------------------------------------------------

$include (c8051f200.inc)   ; Include regsiter definition file

;-----------------------------------------------------------------------------
; EQUATES
;-----------------------------------------------------------------------------

;Port equates

PLTCURP		equ		p0.0		;plate current sense
HVSENSEP	equ		P0.1		;HV voltage sense
PFIP			equ		p0.2		;power failure input (48V sense)
BIASSENP	equ		p0.3		;bias voltage sense
G2SENP		equ		p0.4		;G2 supply sense
TEMP   		equ		p0.5		;temperature sensor input
;					equ		p0.6		;unused
;					equ		p0.7		;unused	

;					equ		p1.0		;unused
;					equ		p1.1		;unused
;					equ		p1.2		;unused
;					equ		p1.3		;unused
;					equ		p1.4		;unused
;					equ		p1.5		;unused
;					equ		p1.6		;unused
;					equ		p1.7		;unused

HVON			equ		p2.0		;output bit, to enable high voltage
FILON			equ		p2.1		;output bit, to enable PA filament power
REVERSE		equ		p2.2		;output bit, toreverse PA filament polarity
;					equ		p2.3		;unused
;					equ		p2.4		;unused
LCD_RS		equ		p2.5		;output bit, LCD module register select (RS)
LCD_RW		equ		p2.6		;output bit, LCD module read/!write (RW)
LCD_E			equ		p2.7		;output bit, LCD module enable (E)

LCD_DATA	equ		p3			;output port, LCD module data port



; RAM usage - these byte locations in RAM are used for variable storage

DISP_BUF	equ		70h			;16-byte display buffer, at top of low RAM
TICK			equ		6fh			;decremented for display update timer
ADCLO			equ		6eh			;ADC converted /2, low byte
ADCHI			equ		6dh			;ADC converted /2, hi byte
BCDLO			equ		6ch			;ADC converted to BCD, low byte
BCDHI			equ		6bh			;ADC converted to BCD, hi nyte
DSPCNT		equ		6ah			;display counter used to sequence display messages


; Bit equates - these bit locations in RAM are used for flags

VIDISP		bit		0				;flag to indicate when display is to be updated



; ADC input equates - see ADCONVERT routine for definition - used to select
; which ADC input to sample

ADPLTI		equ		0				;plate current sense, port 0 bit 0
ADPLTV		equ		1				;plate voltage sense, port 0 bit 1
ADPFI			equ		2				;power fail input, port 0 bit 2
ADBIAS		equ		3				;bias voltage sense, port 0 bit 3
ADG2			equ		4				;G2 voltage sense, port 0 bit 4
ADTEMP		equ		5				;temperature sensor voltage /2, port 0 bit 5


; Limit equates - values are in mV at port pin

ACLO_LLIM	equ		1000		;low limit for line voltage when operating
ACOK_LLIM	equ		1500		;minimum line voltage to re-start after low line

BIAS_HLIM	equ		1280		;bias must be less than this limit (it's negative)

G2_LLIM		equ		1200		;low limit for G2 sense
G2_HLIM		equ		1800		;high limit for G2 sense

HV_LLIM		equ		700			;low limit for HV
HV_HLIM		equ		900			;high limit for HV

OC_HLIM		equ		1700		;maximum allowed plate current

TEMP_HLIM	equ		1865		;max internal temp (=100 C)



; Other equates

WARM_TIME	equ		30			;time in seconds for filament warmup, must be <99



;-----------------------------------------------------------------------------
; RESET and INTERRUPT VECTORS
;-----------------------------------------------------------------------------

          cseg 	AT 0



	        ljmp 	INIT    					;reset vector			
                               

					cseg	AT 0bh
					ljmp	TINT							;timer 0 interrupt

;-----------------------------------------------------------------------------
; CODE SEGMENT
;-----------------------------------------------------------------------------


AMPCTRL   segment  CODE

          rseg     AMPCTRL        ; Switch to this code segment.
          using    0              ; Specify register bank for the
                                  ; following program code.

; INIT - Initialize the processor, ports, etc.
;

INIT:     mov   WDTCN, #0DEh			; Disable the WDT. 
          mov   WDTCN, #0ADh

; initialize the IO ports.  

					mov		PRT0MX,#0					;no special functions on P0
					mov		PRT1MX,#0					;no special functions on P1
					mov		PRT2MX,#01000000b	;no special P2, enbl weak PU's xcept p3

					mov		P0,#0ffh					;initialize port 0 to all ones (inputs)
					mov		PRT0CF,#0					;port 0 config open-drain (inputs)
					mov		P0MODE,#0					;port 0 mode analog inputs

					mov		P1,#0							;initialize port 1
					mov		PRT1CF,#0					;port 1 config open-drain
					mov		P1MODE,#0ffh			;port 1 digital inputs enabled

					mov		P2,#0							;initialize port 1
					mov		PRT2CF,#0ffh			;port 2 config push-pull
					mov		P2MODE,#0ffh			;port 2 digital inputs enabled

					mov		P3,#0							;initialize port 1
					mov		PRT3CF,#0ffh			;port 3 config push-pull
					mov		P3MODE,#0ffh			;port 3 digital inputs enabled

; initialize the stack to high half of memory

					mov		SP,#80h						;initialize the stack pointer to high ram


; Clock init - set up to use the 11.059MHz crystal insted of the internal osc
; Initially the processor strts up using an internal 2MHz oscillator
; This procedure is as reccomended by Cygnal

CLKINIT:	mov		OSCXCN,#01100111b	;turn on crystal oscillator

					mov		r6,#5
CLKDL1:		mov		r7,#0
					djnz	r7,$
					djnz	r6,CLKDL1					;wait at least 1mS before reading status

					mov		a,OSCXCN					;oscillator status
					jnb		acc.7,$						;hang here until oscillator valid flag

					orl		OSCICN,#00001000b	;select 11.059 xtal as clock source


; TINIT - set up timer 0 for a 50mS periodic interrupt, timer 1 for delay use
; At 11.059MHz clock. clk/12 to timer gives 1.085uS/count

TINIT:		mov	TMOD,#00010001b			;timer 0 and 1 in mode 1, 16-bit
					mov	CKCON,#00000000b		;timers use clock/12 as input

					mov	TH0, #HIGH(-46079)	;set timer 0 for 50mS
					mov	TL0,#LOW(-46079)

					mov	TCON,#00010000b			;enable timer 0

					mov	IE,#10000010b				;enable timer 0 interrupt, and no others

					mov	TICK,#20						;init display tick for 1S update


; ADCINIT - initialize the ADC

ADCINIT:	mov		REF0CN,#00000000b	;select external reference pin
					mov		ADC0CF,#01100000b	;set ADC clock = sysclk/8, PGA gain = 1
					mov		ADC0CN,#10000000b	;enable ADC, data right justified



; LCDINIT - initializes LCD display per Optrex's initialization instructions
; This is written for a 16x1 character, 1/16 duty display like the DMC-16128 series
; Note that this display is addressed as if it were a 8x2 display

LCDINIT:		clr		LCD_E					;enable inactive
		
						mov		a,#20
						call	delay					;20mS delay for LCD to initialize

						mov		a,#00110000b	;8-bit interface command
						call	LCD_CMD				;send it

						mov		a,#5
						call	delay					;wait 5ms
	
						mov		a,#00110000b	;8-bit interface command
						call	LCD_CMD				;send it

						call	wait1ms				;wait 1ms

						mov		a,#00110000b	;8-bit interface command
						call	LCD_CMD				;send it

						call	wait1ms				;wait for command to complete

						mov		a,#00111000b	;function set 1/16, 5x8
						call	LCD_CMD				;write it

						call	wait1ms				;wait for command to complete

						mov		a,#00001000b	;display off
						call	LCD_CMD				;write it

						call	wait1ms				;wait for command to complete

						mov		a,#00000001b	;clear display
						call	LCD_CMD				;write it

						mov		a,#20
						call	delay					;20mS delay for clear command to complete

						mov		a,#00000110b	;set entry mode, inc addr, no shift
						call	LCD_CMD				;write it

						call	wait1ms				;wait for command to complete

						mov		a,#00001100b	;display on, no cursor or blink
						call	LCD_CMD				;write it

						call	wait1ms				;wait for command to complete


; Read flash at address 2000h to determine next state of filament reversal
; then store new (inverted) byte into flash
; This reverses the direction of filament current via a relay each power-up cycle

FILREV:			mov		DPTR,#2000h		;flash address in top block
						clr		a							;set offset from DPTR to 0
						movc	a,@a+DPTR			;get the byte from flash

						mov		c,acc.0				;set port bit to match flash
						mov		REVERSE,c

						cpl		a							;new value to write to flash

						mov		FLSCL,#10001000b		;set flash memory timing
						mov		PSCTL,#00000011b		;enable flash erase

						movx	@DPTR,a				;dummy write to erase block

						mov		PSCTL,#00000001b		;enable flash write

						movx	@DPTR,a				;write the new data to flash

						mov		PSCTL,#0			;disable flash writes


; STARTUP - start the process of turning the amplifier on

STARTUP:		jnb		VIDISP,$			;wait for display flag (~1S after reset)
						clr		VIDISP				;clear flag
						
						mov		a,#(START_STR-TEXT)	;startup string
						call	DISPLAY				;display startup string

						jnb		VIDISP,$			;wait 1S for display flag
						clr		VIDISP				;clear flag

; check to see if line voltage is up high enough
						
						call	LINEOK
						jnc		WARMUP				;line voltage is OK, start warmup


; LOWLINE - line voltage is too low, display message and wait for it to rise

LOWLINE:		clr		HVON					;shut off high voltage, if it was on
						clr		FILON					;shut off PA filament, if it was on
						
						mov		a,#(LOWLINE_STR-TEXT)
						call	DISPLAY				;display low line message

						jnb		VIDISP,$			;wait 1S for display flag
						clr		VIDISP				;clear flag

						call	LINEUP				;check and see if line is up yet once/second
						jc		LOWLINE				;loop forever until line is OK


; WARMUP - turn on filaments and start warmup countdown

WARMUP:			mov		a,#(WARMUP_STR-TEXT)
						call	DISPLAY				;display warmup string minus time countdown

						setb	FILON					;filaments on

						mov		r6,#WARM_TIME	;get warmup time

WARMLOOP:		mov		a,r6					;get time remaining

; convert time remaining into a 2-digit (00-99) BCD number

						mov		b,#10	
						div		ab						;divide by 10
						swap	a							;move result to high nibble
						orl		a,b						;OR In Remainder

						mov		r5,a					;save BCD value temporarily

; convert the BCD to ASCII characters, and insert into the display buffer

						anl		a,#0f0h				;strip lower nibble
						swap	a							;put high digit in lower nibble of a
						add		a,#'0'				;add ascii 0 (zero)
						mov		DISP_BUF+11,a	;write lower digit into display buffer

						mov		a,r5					;get BCD value back
						anl		a,#0fh				;strip upper nibble
						add		a,#'0'				;add ascii 0 (zero)
						mov		DISP_BUF+12,a	;write upper digit into display buffer

						call	WRTLCD				;display the combined warmup string and countdown

						jnb		VIDISP,$			;wait 1S for display flag
						clr		VIDISP				;clear flag

						call	LINEOK				;check and see if line voltage still OK once each second
						jc		LOWLINE				;if line is low go to low line routine

						djnz	r6,WARMLOOP		;loop until counted down to 0

; check to see if bias is OK before HV on

						call	CHKBIAS				;check bias, if it is not will die (shut down and hang)

; enable HV, and monitor plate current (only) for overcurrent for one second

						setb	HVON					;HV on

						mov		a,#(HVON_STR-TEXT)
						call	DISPLAY				;display HV on message

HVUP:				call	CHKOC					;check for overcurrent, if error will die

						jnb		VIDISP,HVUP		;wait 1S for HV to come up, while checking for overcurrent
						clr		VIDISP				;clear the flag

						mov		DSPCNT,#4			;initialize display message counter

; at this point HV has been anabled for 1 second and we can check voltages


; MAIN - main program loop, checks parameters constantly, and updates voltage/current
; display once each second

MAIN:				call	LINEOK				;check and see if line voltage still OK
						jc		LOWLINE				;if low go to low line routine

						call	CHKBIAS				;check bias, if error will die

						call	CHKG2					;check G2 supply, if error will die

						call	CHKHV					;check HV, if error will die

						call	CHKOC					;check for overcurrent, if error will die

						call	CHKTEMP				;check temp, if too hot die

						jnb		VIDISP,MAIN		;wait for fdisplay flag to updatev I/V display
						clr		VIDISP				;clear the flag

						djnz	DSPCNT,DSPRUN	;if not time to display temp

; Display internal temperature
;	Sensor outputs 10mV/degree K, which is divided by 2 befor fed to ADC pin


DSPTEMP:		mov		DSPCNT,#4			;reset display counter so temp displayed every 4 sec

						mov		a,#(TEMP_STR-TEXT)	;text for temp display,minus the numbers
						call	DISPLAY

; read the temp sensor voltage and convert it to BCD

						mov		a,#ADTEMP			;read temp sensor
						call	ADCONVERT			;convert it
						
; Subtract 273 degrees K, or 1.365V
; Note that this fails if below 0 degrees C!

						clr		c

						mov		a,ADCLO				;low byte ADC
						subb	a,#LOW(1365)
						mov		ADCLO,a				;temp save

						mov		a,ADCHI
						subb	a,#HIGH(1365)
						mov		ADCHI,a				;temp save


; Now divide by 5 to get degreec C, by multiplying by 2 then throwing away low digit

						clr		c	

						mov		a,ADCLO
						rlc		a
						mov		ADCLO,a				;LSB times two

						mov		a,ADCHI
						rlc		a
						mov		ADCHI,a				;MSB times two
						
; Convert to BCD and display temperature

						call	ADCBCD				;convert to BCD

						mov		a,BCDHI				;msb of BCD temperature
						anl		a,#0f0h				;strip lower nibble
						swap	a

						jnz		NOLZEROT			;not a leading zero

; supress the leading zero (10 - 99 degrees) by writing a space instead

						mov		a,#" "				;supress leading zero, write a space
						jmp		LZEROT

NOLZEROT:		add		a,#"0"				;add ascii zero

LZEROT:			mov		DISP_BUF+6,a	;put 100's digit temperature in display buffer

						mov		a,BCDHI				;msb of BCD plate current
						anl		a,#0fh				;strip upper nibble
						add		a,#"0"				;add ascii zero to get ASCII byte for the digit
						mov		DISP_BUF+7,a	;put 10's digit temperature in display buffer

						mov		a,BCDLO				;lsb
						anl		a,#0f0h				;strip lower
						swap	a
						add		a,#"0"				;add ascii zero to get ASCII byte for the digit
						mov		DISP_BUF+8,a	;put 1's digit temperature in display buffer
								
						call	WRTLCD				;send buffer to display

						jmp		MAIN


; Run display
; update the plate voltage and current display (done once each seconds)

DSPRUN:			mov		a,#(RUN_STR-TEXT)	;text for I/V display,minus the numbers
						call	DISPLAY

; first, read the plate voltage and convert it to BCD

						mov		a,#ADPLTV			;read plate voltage
						call	ADCONVERT			;convert it

						call	ADCBCD				;convert ADC value to BCD digits


						mov		a,BCDHI				;msb of BCD voltage
						anl		a,#0f0h				;strip lower nibble
						swap	a
						jnz		NOLZEROV			;highest digit is not a 0 (voltage between 1000 and 2047)

; supress the leading zero (100 - 999 volts) by writing a space instead

						mov		a,#" "				;supress leading zero, write a space
						jmp		LZEROV

NOLZEROV:		add		a,#"0"				;add ascii zero to get ASCII byte for the digit

LZEROV:			mov		DISP_BUF,a 		;put 1000's digit voltage in display buffer

						mov		a,BCDHI				;msb of BCD voltage
						anl		a,#0fh				;strip upper nibble
						add		a,#"0"				;add ascii zero to get ASCII byte for the digit
						mov		DISP_BUF+1,a	;put 100's digit in display buffer

						mov		a,BCDLO				;lsb of BCD voltage
						anl		a,#0f0h				;strip lower nibble
						swap	a
						add		a,#"0"				;add ascii zero to get ASCII byte for the digit
						mov		DISP_BUF+2,a	;put 10's digit in display buffer
								
						mov		a,BCDLO
						anl		a,#0fh		
						add		a,#"0"				;add ascii zero to get ASCII byte for the digit
						mov		DISP_BUF+3,a	;put 1's digit in display buffer

; now, read the plate current and convert it to BCD

						mov		a,#ADPLTI			;read plate current
						call	ADCONVERT			;convert it
						call	ADCBCD				;convert to BCD

						mov		a,BCDHI				;msb of BCD plate current
						anl		a,#0f0h				;strip lower nibble
						swap	a

						jnz		NOLZEROI			;not a leading zero

; supress the leading zero (10 - 99 mA) by writing a space instead

						mov		a,#" "				;supress leading zero, write a space
						jmp		LZEROI

NOLZEROI:		add		a,#"0"				;add ascii zero

LZEROI:			mov		DISP_BUF+9,a	;put 100's digit plate current in display buffer

						mov		a,BCDHI				;msb of BCD plate current
						anl		a,#0fh				;strip upper nibble
						add		a,#"0"				;add ascii zero to get ASCII byte for the digit
						mov		DISP_BUF+10,a	;put 10's digit plate current in display buffer

						mov		a,BCDLO				;lsb
						anl		a,#0f0h				;strip lower
						swap	a
						add		a,#"0"				;add ascii zero to get ASCII byte for the digit
						mov		DISP_BUF+11,a		;put 1's digit plate current in display buffer
								
						call	WRTLCD				;send buffer to display


						jmp		MAIN					;repeat forever until power off or an error



;-----------------------------------------------------------------------------
; SUBROUTINES
;-----------------------------------------------------------------------------




; LINEUP - check and see if line voltage is above minimum to turn back on
; Called with - nothing
; Uses - 
; Returns - C is set if line not OK, clear otherwise

LINEUP:			mov		a,#ADPFI
						call	ADCONVERT			;read the PFI input with ADC

						clr		c

						mov		a,ADCLO				;get ADC low byte
						subb	a,#LOW(ACOK_LLIM)	;subtract the low byte of the ACOK low limit

						mov		a,ADCHI				;get ADC hi btye
						subb	a,#HIGH(ACOK_LLIM)	;subtract high byte - if carry, under limit

						ret


; LINEOK - check and see if line voltage is above minimum to operate
; Called with - nothing
; Uses - 
; Returns - C is set if line not OK, clear otherwise

LINEOK:			mov		a,#ADPFI
						call	ADCONVERT			;read the PFI input with ADC

						clr		c

						mov		a,ADCLO				;get ADC low byte
						subb	a,#LOW(ACLO_LLIM)	;subtract the low byte of the ACOK low limit

						mov		a,ADCHI				;get ADC hi btye
						subb	a,#HIGH(ACLO_LLIM)	;subtract high byte - if carry, under limit

						ret


; CHKBIAS - check and see if bias voltage is above minimum
; Called with - nothing
; Uses - 
; Returns - C is set if line not OK, clear otherwise

CHKBIAS:		mov		a,#ADBIAS
						call	ADCONVERT			;read the PFI input with ADC

						clr		c

						mov		a,#LOW(BIAS_HLIM)	;get the low byte of the bias high limit
						subb	a,ADCLO				;subtract ADC low byte

						mov		a,#HIGH(BIAS_HLIM)	;get high byte - if carry, over limit
						subb	a,ADCHI				;subtract ADC hi btye

						jc		BIASERR				;if error

						ret

; BIASERR - bias error handler

BIASERR:		clr		HVON
						mov		a,#(BIASERR_STR-TEXT)	;bias error message
						call	DISPLAY

						jmp		DIE						;fatal error


; CHKOC - check and see if plate current is excessive
; Called with - nothing
; Uses - 
; Returns - C is set if line not OK, clear otherwise

CHKOC:			mov		a,#ADPLTI
						call	ADCONVERT			;read the PFI input with ADC

						clr		c

						mov		a,#LOW(OC_HLIM)	;get the low byte of the bias high limit
						subb	a,ADCLO				;subtract ADC low byte

						mov		a,#HIGH(OC_HLIM)	;get high byte - if carry, over limit
						subb	a,ADCHI				;subtract ADC hi btye

						jc		OCERR				;if error

						ret

; OCERR - bias error handler

OCERR:			clr		HVON
						mov		a,#(OCERR_STR-TEXT)	;overcurrent error message
						call	DISPLAY

						jmp		DIE						;fatal error


; CHKG2 - check and see if G2 voltage is within limits
; Called with - nothing
; Uses - 
; Returns - nothing if OK, dies if error

CHKG2:			mov		a,#ADG2
						call	ADCONVERT			;read the G2 input with ADC

						clr		c

						mov		a,ADCLO				;subtract ADC low byte
						subb	a,#LOW(G2_LLIM)	;get the low byte of the G2 low limit

						mov		a,ADCHI				;subtract ADC hi btye
						subb	a,#HIGH(G2_LLIM)	;get high byte - if carry, under limit

						jc		G2ERR				;if carry, is under low limit

						mov		a,#LOW(G2_HLIM)	;get the low byte of the G2 high limit
						subb	a,ADCLO				;subtract ADC low byte

						mov		a,#HIGH(G2_HLIM)	;get high byte - if carry, over limit
						subb	a,ADCHI				;subtract ADC hi btye

						jc		G2ERR					;if carry, is over high limit

						ret

; G2ERR - bias error handler

G2ERR:			clr		HVON
						mov		a,#(G2ERR_STR-TEXT)	;G2 error message
						call	DISPLAY

						jmp		DIE						;fatal error


; CHKHV - check and see if HV regulator feeedback voltage is within limits
; Called with - nothing
; Uses - 
; Returns - nothing if OK, dies if error

CHKHV:			mov		a,#ADPLTV
						call	ADCONVERT			;read the HV input with ADC

						clr		c

						mov		a,ADCLO				;subtract ADC low byte
						subb	a,#LOW(HV_LLIM)	;get the low byte of the HV low limit

						mov		a,ADCHI				;subtract ADC hi btye
						subb	a,#HIGH(HV_LLIM)	;get high byte - if carry, under limit

						jc		HVERR				;if carry, is under low limit

						mov		a,#LOW(HV_HLIM)	;get the low byte of the HV high limit
						subb	a,ADCLO				;subtract ADC low byte

						mov		a,#HIGH(HV_HLIM)	;get high byte - if carry, over limit
						subb	a,ADCHI				;subtract ADC hi btye

						jc		HVERR					;if carry, is over high limit

						ret

; HVERR - HV regulation error handler

HVERR:			clr		HVON
						mov		a,#(HVERR_STR-TEXT)	;HV reg error message
						call	DISPLAY

						jmp		DIE						;fatal error



; CHKTEMP - check and see if internal temp is excessive
; Called with - nothing
; Uses - 
; Returns - C is set if line not OK, clear otherwise

CHKTEMP:		mov		a,#ADTEMP
						call	ADCONVERT			;read the PFI input with ADC

						clr		c

						mov		a,#LOW(TEMP_HLIM)	;get the low byte of the temp high limit
						subb	a,ADCLO				;subtract ADC low byte

						mov		a,#HIGH(TEMP_HLIM)	;get high byte - if carry, over limit
						subb	a,ADCHI				;subtract ADC hi btye

						jc		TEMPERR				;if error

						ret

; TEMPERR - bias error handler

TEMPERR:		clr		HVON
						mov		a,#(TEMPERR_STR-TEXT)	;overtemp error message
						call	DISPLAY

						jmp		DIE						;fatal error




;DIE - fatal error

DIE:				mov		r7,#5					;wait 5S for filaments off

DIELOOP:		jnb		VIDISP,$			;wait 1S for display flag
						clr		VIDISP				;clear flag

						djnz	r7,DIELOOP

						clr		FILON

						jmp		$							;die


; ADCONVERT - do an A/D conversion
; Called with - desired channel to convert in a; 000 pp bbb where pp = port, bb = bit
; Uses - acc, r0, r1, r3, r4
; Returns - Converted 11-bit binary value in ADCHI (MSB) and ADCLO (LSB)

ADCONVERT:	orl		a,#00100000b	;enable analog mux to selected channel
						mov		AMX0SL,a			;set the mux

						mov		a,#10
						call	delay					;let port mux settle

						setb	ADBUSY				;start dummy conversion
						jb		ADBUSY,$			;hang until conversion complete

; the ADC is read for times, and the average of the 4 values is saved

						mov		r7,#4					;going to read ADC 4 times and average readings

						mov		ADCLO,#0			;clear old reading
						mov		ADCHI,#0

ADCLOOP:		setb	ADBUSY				;start a conversion
						jb		ADBUSY,$			;hang until conversion complete

						mov		a,ADC0L				;get the result of this conversion
						add		a,ADCLO				;sum it with the results of previous conversions
						mov		ADCLO,a				;save result
						
						mov		a,ADC0H				;high byte
						addc	a,ADCHI				;sum results into ADCLO and ADCHI
						mov		ADCHI,a				;save result

						djnz	r7,ADCLOOP		;repeat 4 times

; the value in ADCHI and ADCLO is now the sum of 4 conversions.  We will divide it by
; 8 - 4 to get the average, and 2 more to get the data into milivolts (assuming that
; VREF is 2.048V, we want a 1.000V input to yield a value of 1000)

						mov		r7,#3					;going to divide by 8, by dividing by 2 3 times

ADCDLOOP:		mov		A,ADCHI				;get high ADC
						clr		c							;clear carry
						rrc		a							;rotate right through carry (divide by 2)
						mov		ADCHI,a				;save this

						mov		a,ADCLO				;low ADC
						rrc		a							;divide by 2
						mov		ADCLO,a				;save it

						djnz	r7,ADCDLOOP		;divide by 2 three times

						ret

; ADCBCD - Take ADC and convert into 4-digit BCD
; Called with - nothing, but uses ADCLO and ADCHI
; Uses - a, b
; Returns - updates BCDLO and BCDHI


ADCBCD:			mov		a,ADCLO				;get low ADC /2

						mov		b,#100
						div		ab						;divide By 100
						mov		BCDHI,a				;store high digit (00h - 02h)

						mov		a,b						;get remainder (0-99)
						mov		b,#10	
						div		ab						;divide by 10
						swap	a							;move result to high nibble
						orl		a,b						;OR In Remainder
						mov		BCDLO,a				;store lo digit (00h - 99h)

						mov		a,ADCHI				;get the high byte ADC /2
						jnb		acc.0,NO256		;256 bit not set

						mov		a,BCDLO				;get the low digit
						add		a,#56h				;add the '56' from '256' BCD
						da		a							;correct to BCD
						mov		BCDLO,a				;save it back

						mov		a,BCDHI				;get high digit
						addc	a,#02h				;add the '2' from '256' plus carry
						da		a							;and make into BCD
						mov		BCDHI,a				;save it

NO256:			mov		a,ADCHI				;get the high byte ADC
						jnb		acc.1,NO512		;512 bit not set

						mov		a,BCDLO				;get the low digit
						add		a,#12h				;add the '12' from '512' BCD
						da		a							;correct to BCD
						mov		BCDLO,a				;save it back

						mov		a,BCDHI				;get high digit
						addc	a,#05h				;add the '5' from '512'plus carry
						da		a							;and make into BCD
						mov		BCDHI,a			;save it

NO512:			mov		a,ADCHI				;get thehigh byte ADC
						jnb		acc.2,NO1024	;1024 bit not set

						mov		a,BCDLO				;get the low digit
						add		a,#24h				;add the '24' from '1024' BCD
						da		a							;correct to BCD
						mov		BCDLO,a				;save it back

						mov		a,BCDHI				;get high digit
						addc	a,#10h				;add the '10' from '1024'plus carry
						da		a							;and make into BCD
						mov		BCDHI,a			;save it

NO1024:			ret


; DISPLAY - Write a 16-character message to display
; Called with - ROM address offset in a
; Uses - r0, r1, r7, acc
; Returns - nothing

					
DISPLAY:		mov		r1,a					;save the pointer to the message
						mov		DPTR,#TEXT		;init data pointer to text

						mov		r0,#(DISP_BUF+0fh)	;set r0 to end of disp buffer
						mov		r7,#10h				;init r7 for  16 characters
						
DSPLOOP:		mov		a,r1					;get beginning of text string addr
						add		a,r7					;add loop count
						dec		a							;minus one for loop count

						movc	a,@a+DPTR			;get char from ROM
						mov		@r0,a					;and put it into DSP_BUF

						dec		r0						;move to next char in buffer
						djnz	r7,DSPLOOP		;continue until all 16 written

						call	WRTLCD

						ret


; LCD_CMD - Writes a command to the LCD.  Does not wait for LCD ready.

; Called with - command to be written in acc
; Uses - acc, ports
; Returns - nothing

LCD_CMD:		mov		LCD_DATA,a		;data to LCD port
						clr		LCD_RS				;select command register
						clr		LCD_RW				;write
						nop									;ensure setup time

						setb	LCD_E					;send to LCD

						nop
						nop
						nop
						nop									;make enable pulse >450nS

						clr		LCD_E

						nop
						nop
						nop
						nop

						ret


; LCD_WR - Writes data to the LCD at the current address.  Waits for LCD ready.

; Called with - data to be written in acc
; Uses - acc, ports
; Returns - nothing

LCD_WR:			mov		LCD_DATA,a		;data to LCD port
						setb	LCD_RS				;select data register
						clr		LCD_RW				;write

						nop									;ensure setup time

						setb	LCD_E					;send to LCD

						nop
						nop
						nop
						nop									;make enable pulse >450nS

						clr		LCD_E

						nop
						nop
						nop
						nop

						call	wait1ms			;wait for completion

						ret


; WRTLCD - Write LCD display
; Called with - Character data in 16 bytes of RAM starting at DISP_BUF
; Uses - acc, r0, r7
; Returns - nothing

WRTLCD:			mov		a,#10000000b	;set DD ram address 0
						call	LCD_CMD				;write command

						call	wait1ms				;wait for command to complete

						mov		r0,#DISP_BUF	;init r0 with addr of disp buf
						mov		r7,#8					;first section of 8 characters

FIRST8:			mov		a,@r0					;get the character from the buf
						call	LCD_WR				;write it to the LCD

						inc		r0						;point to next character

						djnz	r7,FIRST8			;loop until first 8 char written

						mov		a,#11000000b	;set DD ram address 40
						call	LCD_CMD				;write command

						call	wait1ms

						mov		r7,#8					;second section of 8 characters
	
LAST8:			mov		a,@r0					;get the character from the buf
						call	LCD_WR				;write it to the LCD

						inc		r0						;point to next character

						djnz	r7,LAST8			;loop until first 8 char written

						ret


; DELAY - delay some number of mS (1-256)
; Called with - desired delay in acc
; Uses - acc, r7, timer 1
; Returns - nothing

DELAY:			clr		TR1						;stop timer in case it's running
						clr		TF1						;clear timer flag if set
						mov		r7,a					;time count in r7

DLYLOOP:		mov		TH1,#HIGH(-922)
						mov		TL1,#LOW(-922)	;set timer for 1mS

						setb	TR1						;start the timer

						jnb		TF1,$					;hang until timer times out
				
						clr		TR1						;stop the timer
						clr		TF1						;clear timer flag

						djnz	r7,DLYLOOP		;interate for each mS

						ret


; WAIT1MS - delay one mS 
; Called with - nothing
; Uses - timer 1
; Returns - nothing

WAIT1MS:		clr		TR1						;stop timer in case it's running
						clr		TF1						;clear timer flag if set

WAITLOOP:		mov		TH1,#HIGH(-922)
						mov		TL1,#LOW(-922)	;set timer for 1mS

						setb	TR1						;start the timer

						jnb		TF1,$					;hang until timer times out
				
						clr		TR1						;stop the timer
						clr		TF1						;clear timer flag

						ret



;-----------------------------------------------------------------------------
; INTERRUPT SERVICE ROUTINES
;-----------------------------------------------------------------------------


; Timer interrupt
; Entered every 50mS
; Uses no registers other than timers, TICK, and VIDISP bit


TINT:				clr		TR0							;stop the timer

						djnz	TICK,TINTRET		;decrement tick, if not update time ret
	
						mov		TICK,#20				;reset update count for 2S
	
						setb	VIDISP					;set flag for voltage/current display update

; At 11.059MHz clock. clk/12 to timer gives 1.085uS/count

TINTRET:		mov		TH0, #HIGH(-46079)	;reset for 10mS
						mov		TL0,#LOW(-46079)

						clr		TF0							;clear the timer flag
						setb	TR0							;restart the timer

						reti



;-----------------------------------------------------------------------------
; ROM DATA
;-----------------------------------------------------------------------------

; TEXT - text strings to write to display - each is 16 characters long

TEXT:																		;Beginning of text section

START_STR:		db 	"813 amp rev 1.1 "		
LOWLINE_STR:	db	"Low line voltage"
WARMUP_STR:		db	"Warming up   ..."
RUN_STR:			db	"     V       mA "
TEMP_STR:			db	"Temp:     Deg. C"
HVON_STR:			db	"  Enabling B+   "
BIASERR_STR:	db	"ERROR: BIAS FAIL"
G2ERR_STR:		db	" ERROR: G2 FAIL "
HVERR_STR:		db	"ERROR: PLT VOLT "
OCERR_STR:		db	"ERROR: PLT CURR "
TEMPERR_STR:	db	"ERROR: OVERTEMP "




;-----------------------------------------------------------------------------
; End of file.


END
